function fmtTime(ms) {
  return new Date(ms).toLocaleString([], {
    weekday: "short",
    month: "short",
    day: "numeric",
    hour: "numeric",
    minute: "2-digit",
  });
}

function minutesUntil(startMs) {
  const diffMs = startMs - Date.now();
  if (diffMs <= 0) return 0;
  return Math.ceil(diffMs / 60000);
}

function decodePayloadFromQuery() {
  const params = new URLSearchParams(location.search);
  const payloadB64 = params.get("payload");
  if (!payloadB64) return null;

  try {
    // reminder window encodes: btoa(unescape(encodeURIComponent(JSON.stringify(payload))))
    // Here we decode robustly back to UTF-8.
    const binaryStr = atob(payloadB64);
    const bytes = new Uint8Array(binaryStr.length);
    for (let i = 0; i < binaryStr.length; i++) bytes[i] = binaryStr.charCodeAt(i);

    let jsonStr = "";
    if (typeof TextDecoder !== "undefined") {
      jsonStr = new TextDecoder("utf-8").decode(bytes);
    } else {
      // Fallback (older environments)
      jsonStr = decodeURIComponent(
        Array.prototype.map
          .call(binaryStr, (c) => "%" + ("00" + c.charCodeAt(0).toString(16)).slice(-2))
          .join("")
      );
    }

    return JSON.parse(jsonStr);
  } catch (e) {
    console.error("Failed to decode payload:", e);
    return null;
  }
}

async function safeSendMessage(message) {
  try {
    return await chrome.runtime.sendMessage(message);
  } catch (e) {
    console.warn("sendMessage failed:", e);
    return null;
  }
}

async function init() {
  const payload = decodePayloadFromQuery();
  if (!payload) return;

  const titleEl = document.getElementById("title");
  const timeEl = document.getElementById("timeText");
  const joinWrap = document.getElementById("joinWrap");
  const joinLink = document.getElementById("joinLink");
  const joinBtn = document.getElementById("joinBtn");
  const snoozeBtn = document.getElementById("snoozeBtn");
  const snoozeSelect = document.getElementById("snoozeSelect");
  const dismissBtn = document.getElementById("dismissBtn");

  if (!titleEl || !timeEl || !joinWrap || !joinLink || !joinBtn || !snoozeBtn || !snoozeSelect || !dismissBtn) {
    console.error("Reminder UI missing elements");
    return;
  }

  const startMs = Number(payload.startMs);
  const safeStartMs = Number.isFinite(startMs) ? startMs : Date.now();

  // ---------- Populate UI ----------
  titleEl.textContent = payload.title || "Meeting";

  function renderTimeLine() {
    const mins = minutesUntil(safeStartMs);
    timeEl.textContent = `${fmtTime(safeStartMs)} — in ${mins} minute${mins === 1 ? "" : "s"}`;
  }

  renderTimeLine();

  const joinUrl = typeof payload.joinUrl === "string" ? payload.joinUrl : "";
  const joinPreview = payload && payload.joinPreview ? payload.joinPreview : null;

  if (joinUrl) {
    joinWrap.style.display = "block";

    const label =
      (joinPreview && (joinPreview.label || joinPreview.provider)) ||
      "Join meeting";

    joinLink.textContent = label;
    joinLink.href = joinUrl;

    const provider = (joinPreview && joinPreview.provider) ? joinPreview.provider : "";
    joinBtn.textContent = (`Join ${provider}`).trim() || "Join";
    joinBtn.disabled = false;
  } else {
    joinWrap.style.display = "none";
    joinBtn.disabled = true;
    joinBtn.textContent = "Join";
    joinLink.textContent = "";
    joinLink.href = "#";
  }

  // ---------- Live countdown + auto-close ----------
  // Update countdown every second; auto-close once meeting starts (with 5 min grace).
  const COUNTDOWN_INTERVAL_MS = 1000;
  const AUTOCLOSE_GRACE_MS = 5 * 60 * 1000;
  const openedAfterStart = Date.now() >= safeStartMs;

  let intervalId = null;

  function cleanupAndClose() {
    if (intervalId) {
      clearInterval(intervalId);
      intervalId = null;
    }
    window.close();
  }

  intervalId = setInterval(() => {
    renderTimeLine();
    if (!openedAfterStart && Date.now() >= safeStartMs + AUTOCLOSE_GRACE_MS) {
      cleanupAndClose();
    }
  }, COUNTDOWN_INTERVAL_MS);

  window.addEventListener("beforeunload", () => {
    if (intervalId) clearInterval(intervalId);
    intervalId = null;
  });

  // ---------- Button Actions ----------
  snoozeBtn.addEventListener("click", async () => {
    const sel = snoozeSelect.value;

    let snoozeUntilMs;
    if (sel === "AT_START") {
      snoozeUntilMs = Math.max(Date.now(), safeStartMs);
    } else {
      const minutes = Number(sel);
      const m = Number.isFinite(minutes) ? minutes : 10;
      snoozeUntilMs = Date.now() + m * 60000;
    }

    await safeSendMessage({
      type: "SNOOZE",
      eventKey: payload.eventKey,
      snoozeUntilMs,
      payload,
    });

    cleanupAndClose();
  });

  dismissBtn.addEventListener("click", async () => {
    await safeSendMessage({
      type: "DISMISS",
      eventKey: payload.eventKey,
    });

    cleanupAndClose();
  });

  joinBtn.addEventListener("click", async () => {
    if (joinUrl) {
      await safeSendMessage({
        type: "OPEN_JOIN",
        url: joinUrl,
      });
    }
    cleanupAndClose();
  });

  // ---------- Keyboard UX ----------
  document.addEventListener("keydown", (e) => {
    // ESC closes window
    if (e.key === "Escape") {
      cleanupAndClose();
      return;
    }

    // Cmd/Ctrl + Enter = Join
    if (e.key === "Enter" && (e.metaKey || e.ctrlKey)) {
      if (!joinBtn.disabled) joinBtn.click();
      return;
    }

    // Enter = Snooze (but don't interfere if user is on a button)
    if (e.key === "Enter" && !e.metaKey && !e.ctrlKey) {
      const active = document.activeElement;
      const isButton = active && active.tagName === "BUTTON";
      if (!isButton) snoozeBtn.click();
    }
  });

  // ---------- Autofocus ----------
  requestAnimationFrame(() => {
    snoozeSelect.focus();
  });
}

init();
